<?php

namespace App\Http\Controllers\Api;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Attendance;
use App\Models\Meeting;
use App\Models\Church;
use App\Models\User;
use Carbon\Carbon;
use App\Models\PrayerLog;
use App\Models\PrayerPoint;
use Mail; 
use DB; 
use Illuminate\Validation\Rule;
use Laravel\Sanctum\PersonalAccessToken;


class UserController extends Controller
{

    public function UserLogin(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'email_or_phone' => [
                'required',
                function ($attribute, $value, $fail) {
                    if (!filter_var($value, FILTER_VALIDATE_EMAIL) && !preg_match('/^\d{10,12}$/', $value)) {
                        $fail('The email or phone number must be a valid email or a phone number with 10-12 digits.');
                    }
                }
            ],
            'password' => 'required',
            'role_type' => 'required|in:user,leader', 
        ]);
    
         if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
    
        $validatedData = $validator->validated();
    
        $user = User::where('email', $validatedData['email_or_phone'] ?? null)
            ->orWhere('phone_no', $validatedData['email_or_phone'] ?? null)
            ->first();
    
        if (!$user || !Hash::check($validatedData['password'], $user->password)) {
            return response()->json([
                'status' => 401,
                'message' => 'Invalid email, phone number, or password',
            ], 401);
        }
    
        // Role-based authorization
        if ($validatedData['role_type'] == 'user' && $user->user_role !== 'Prayer Warrior') {
            return response()->json([
                'status' => 403,
                'message' => 'Unauthorized access. This user is not authorized with User role.',
            ], 403);
        }
    
        if ($validatedData['role_type'] == 'leader' && 
            !in_array($user->user_role, ['Pastor', 'Prayer (Leader)', 'Senior Prayer Warrior (Leader)'])) {
            return response()->json([
                'status' => 403,
                'message' => 'Unauthorized access. This users is not authorized with Leader role.',
            ], 403);
        }
    
        if ($user->user_status == 'Inactive User') {
            return response()->json([
                'status' => 403,
                'message' => 'Your Account is Inactive, please contact the administrator.',
            ], 403);
        }
    
        // Generate authentication token
        $token = $user->createToken($user->email . 'Token')->plainTextToken;
        $church_name = $user->church ? $user->church->church_name : null;
        $church_logo = $user->church ? asset($user->church->church_logo) : null;
    
        return response()->json([
            'status' => 200,
            'message' => 'Login successful',
            'user' => [
                'id'=>$user->id,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'phone_no' => $user->phone_no,
                'email' => $user->email,
                'user_role' => $user->user_role,
                'user_status' => $user->user_status,
                'church_name' => $church_name,
                'profile_photo' => asset($user->profile_photo),
                'church_logo' => $church_logo,
            ],
            'token' => $token,
        ], 200);
    }

    public function ViewProfile(Request $request)
    {
        try {
            $user = Auth::user();
    
            if (!$user) {
                return response()->json([
                    'status' => 401,
                    'message' => 'Unauthorized: User not authenticated',
                ], 401);
            }
    
            return response()->json([
                'status' => 200,
                'message' => 'Profile retrieved successfully',
                'user' => [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' =>  $user->profile_photo 
                    ? asset($user->profile_photo) 
                    : asset('user_profile/profile.png')
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                 'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function UpdateProfile(Request $request)
    {
        try {
            $user = Auth::user();
    
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
    
            // Validate request data
            $validator = Validator::make($request->all(), [
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'phone_no' => 'required|digits:10|unique:users,phone_no,' . $user->id,
            ]);
    
             if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
    
            // Update user profile using validated data
            $user->update($validator->validated());

                     // Update user profile
                $user->first_name = $request->input('first_name');
                $user->last_name = $request->input('last_name');
                $user->phone_no = $request->input('phone_no');
                $user->save();
         
    
            return response()->json([
                'status' => 200,
                'message' => 'Profile updated successfully',
                'user' => [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' => $user->profile_photo 
                    ? asset($user->profile_photo) 
                    : asset('user_profile/profile.png')
                ],
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
              'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function AddUser(Request $request)
    {
        try {
             $allowedRoles = ['Prayer Warrior', 'Pastor', 'Prayer (Leader)', 'Senior Prayer Warrior (Leader)'];
                $allowedStatuses = ['Prayer Warrior', 'Champion Warrior', 'Elite Warrior', 'Inactive User'];
        
                // Manually check if user_role is valid
                if (!in_array($request->user_role, $allowedRoles)) {
                    return response()->json([
                        'status' => 422,
                        'message' => 'The selected role is invalid',
                    ], 422);
                }
        
                // Manually check if user_status is valid
                if (!in_array($request->user_status, $allowedStatuses)) {
                    return response()->json([
                        'status' => 422,
                        'message' => 'The selected status is invalid',
                    ], 422);
                }
        
                // Validate request data
                $validator = Validator::make($request->all(), [
                    'first_name' => 'required|string|max:255',
                    'last_name' => 'required|string|max:255',
                    'phone_no' => 'required|digits:10|unique:users,phone_no',
                    'email' => 'required|email|unique:users,email',
                    // 'user_role' => ['required', Rule::in($allowedRoles)], 
                    // 'user_status' => ['required', Rule::in($allowedStatuses)], 
                    'password' => 'nullable|string|min:6', 
                ]);
            
                
             if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
            
            $user = Auth::user(); 
            $church_id = $user->church_id;
            $addedBy = $user->id;
           
           $randomChars = Str::random(4); 
           $specialChars = '!@#$%^&*()'; 
           $randomSpecialChars = substr(str_shuffle($specialChars), 0, 4); 
           $uniquePassword = 'church' . $randomChars . $randomSpecialChars . 'user';

            // Hash the password before saving
            $hashedPassword = Hash::make($uniquePassword);
        
            $userCount = User::where('church_id', $church_id)->count();
    
             if ($userCount >= 15) {
                return response()->json([
                    'status' => 409,
                    'message' => 'You cannot add more than 15 members to this church',
                ], 400);
            }

            // Create user with default password if not provided
            $user = User::create([
                'first_name' => $request->input('first_name'),
                'last_name' => $request->input('last_name'),
                'phone_no' => $request->input('phone_no'),
                'email' => $request->input('email'),
                'user_role' => $request->input('user_role'),
                'user_status' => $request->input('user_status'),
                'church_id'=>$church_id,
                'added_by'=>$addedBy,
                'password' => $hashedPassword, 
            ]);
        
            if ($user) {
                // Send Email to User
                Mail::send('emails.invitation', ['email' => $user->email, 'password' => $uniquePassword], function ($message) use ($user) {
                    $message->to($user->email);
                    $message->subject('Invitation to Join Church');
                });
            }
                    
            return response()->json([
                'status' => 201,
                'message' => 'User added successfully and Invitation is sent to user email address',
                'user' => $user,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
              'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    public function ShowUser(){
        try {
            $loggedInUser = Auth::user();
    
            //$users = User::where('id', '!=', $loggedInUser->id)->get();
            
            $users = User::where('id', '!=', $loggedInUser->id)
            ->where('added_by',$loggedInUser->id)
            ->get();
            
            $formattedUsers = $users->map(function ($user) {
                $profilePhoto = $user->profile_photo 
                ? asset($user->profile_photo) 
                : asset('user_profile/profile.png');

                return [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' => $profilePhoto,
                ];
            });
    
            return response()->json([
                'status' => 200,
                'message' => 'Users retrieved successfully',
                'users' => $formattedUsers, 
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function EditUser(Request $request, $id){

        try {
            $user = User::find($id);
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
            return response()->json([
                'status' => 200,
                'message' => 'User retrieved successfully',
                'user' => [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' =>  $user->profile_photo 
                    ? asset($user->profile_photo) 
                    : asset('user_profile/profile.png')
                ],
            ], 200);
        } catch (\Exception $e) {
           
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function UpdateUser(Request $request, $id)
    {
        try {
            
              $allowedRoles = ['Prayer Warrior', 'Pastor', 'Prayer (Leader)', 'Senior Prayer Warrior (Leader)'];
                $allowedStatuses = ['Prayer Warrior', 'Champion Warrior', 'Elite Warrior', 'Inactive User'];
        
                // Manually check if user_role is valid
                if (!in_array($request->user_role, $allowedRoles)) {
                    return response()->json([
                        'status' => 422,
                        'message' => 'The selected user role is invalid',
                    ], 422);
                }
        
                // Manually check if user_status is valid
                if (!in_array($request->user_status, $allowedStatuses)) {
                    return response()->json([
                        'status' => 422,
                        'message' => 'The selected user status is invalid',
                    ], 422);
                }

              $user = User::find($id);
               if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
            // Validate incoming data
            $validator = Validator::make($request->all(), [
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'phone_no' => 'required|digits:10|unique:users,phone_no,' . $id,
                'email' => 'required|email|unique:users,email,' . $id,
                  'user_role' => ['required', Rule::in($allowedRoles)], 
                  'user_status' => ['required', Rule::in($allowedStatuses)], 
            ]);
    
          if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
            
         
            $user->first_name = $request->input('first_name');
            $user->last_name = $request->input('last_name');
            $user->phone_no = $request->input('phone_no');
            $user->email = $request->input('email');
            $user->user_role = $request->input('user_role');
            $user->user_status = $request->input('user_status');
            $user->save();
    
            return response()->json([
                'status' => 200,
                'message' => 'User updated successfully',
               'user' => [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                ],
            ], 200);
    
        } catch (\Exception $e) {
            \Log::error('Error updating profile picture: ' . $e->getMessage());

            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    public function DeleteUser($id){
     try {
            $user = User::find($id);
    
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
            
             if ($user->profile_photo) {
                $imagePath = public_path($user->profile_photo);
                if (file_exists($imagePath)) {
                    unlink($imagePath);
                }
            }
            $user->delete();
    
            return response()->json([
                'status' => 200,
                'message' => 'User deleted successfully',
            ], 200);
        } catch (\Exception $e) {
            \Log::error('Error updating profile picture: ' . $e->getMessage());

            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
     public function UpdateProfilePic(Request $request)
    {
        try {
            $user = Auth::user();
        
            if (!$user) {
                return response()->json([
                    'status' => 401,
                    'message' => 'Unauthorized: User not authenticated',
                ], 401);
            }
    
            $validator = Validator::make($request->all(), [
                'profile_photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', 
            ]);
    
            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => $validator->errors()->all(), 
                ], 422);
            }
    
            if ($request->hasFile('profile_photo')) {
                $file = $request->file('profile_photo');
                $filename = 'profile_' . uniqid() . '.' . $file->getClientOriginalExtension(); 
    
                $path = public_path('images/user_profile/' . $filename);
                $file->move(public_path('images/user_profile'), $filename); 
    
                $user->profile_photo = 'images/user_profile/' . $filename; 
                $user->save();
    
                return response()->json([
                    'status' => 200,
                    'message' => 'Profile photo updated successfully',
                    'profile_photo' => asset('images/user_profile/' . $filename), 
                ], 200);
            }
    
            return response()->json([
                'status' => 400,
                'message' => 'No file uploaded',
            ], 400);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
        
    public function ViewAllLeader(){
        
        try {
            $loggedInUser = Auth::user();
    
            //$users = User::where('id', '!=', $loggedInUser->id)->get();
            
            $users = User::where('id', '!=', $loggedInUser->id)
            ->where('user_role','!=','Prayer Warrior')
           ->get();
            
    
            $formattedUsers = $users->map(function ($user) {
                $profilePhoto = $user->profile_photo 
                ? asset($user->profile_photo) 
                : asset('user_profile/profile.png');

                return [
                    'id'=>$user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' => $profilePhoto,
                ];
            });
    
            return response()->json([
                'status' => 200,
                'message' => 'Users retrieved successfully',
                'users' => $formattedUsers, 
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }


    public function Forgotpassword(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|exists:users,email',
            ]);
    
            if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
        
    
            // Retrieve user by email
            $email = $request->email;
            $user = User::where('email', $email)->first();
    
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
    
            // Generate OTP
            $otp = mt_rand(1000, 9999);
            $user->otp = $otp;
            $user->save();
    
            // Insert OTP into password_resets table
            DB::table('password_reset_tokens')->updateOrInsert(
                ['email' => $email],
                ['token' => $otp, 'created_at' => Carbon::now()]
            );
    
            // Send OTP email
            Mail::send('emails.otp', ['otp' => $otp], function ($message) use ($email) {
                $message->to($email);
                $message->subject('OTP Verification');
            });
    
            return response()->json([
                'status' => 200,
                'message' => 'OTP sent successfully to your email',
            ], 200);
        } catch (\Exception $e) {
            \Log::error('Error updating profile picture: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
                'message' => $e->getMessage(),
            ], 500);
        }
    }
    
    
   public function VerifyOtp(Request $request)
    {
    try {
        // Validate request input
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
            'otp' => 'required|digits:4',
        ]);

         if ($validator->fails()) {
            $errors = $validator->errors()->all(); 
        
            return response()->json([
                'status' => 422,
                'message' => $errors, 
            ], 422);
        }
        

        // Retrieve OTP from password_reset_tokens table
        $email = $request->email;
        $otp = $request->otp;
        
        $otpRecord = DB::table('password_reset_tokens')
            ->where('email', $email)
            ->where('token', $otp)
            ->first();

        if (!$otpRecord) {
            return response()->json([
                'status' => 400,
                'message' => 'Invalid OTP or expired OTP',
            ], 400);
        }

        return response()->json([
            'status' => 200,
            'message' => 'OTP verified successfully',
        ], 200);

        } catch (\Exception $e) {
              \Log::error('Error updating profile picture: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function ResetPassword(Request $request)
    {
        try {
            
            $validator = Validator::make($request->all(), [
                'email' => 'required|email|exists:users,email',
                'otp' => 'required|digits:4',
                'password' => 'required|min:8|confirmed'
            ]);
    
           if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
    
            // Check if OTP exists and matches
            $passwordReset = DB::table('password_reset_tokens')
                ->where('email', $request->email)
                ->where('token', $request->otp)
                ->first();
    
            if (!$passwordReset) {
                return response()->json([
                    'status' => 400,
                    'message' => 'Invalid OTP or expired OTP',
                ], 400);
            }
    
            // Update the user's password
            $user = User::where('email', $request->email)->first();
            $user->password = Hash::make($request->password);
            $user->save();
    
            // Delete the OTP record
            DB::table('password_reset_tokens')->where('email', $request->email)->delete();
    
            return response()->json([
                'status' => 200,
                'message' => 'Password reset successfully',
            ], 200);
    
        } catch (\Exception $e) {
            \Log::error('Error updating profile picture: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
        
    public function Logout(Request $request)
    {
        try {
            $user = $request->user(); 
            if ($user) {
                $user->currentAccessToken()->delete(); 
            }
            return response()->json([
                'status' => 200,
                'message' => 'Logout successful'
            ], 200);
            
        } catch (\Exception $e) {
            
             \Log::error('Error updating profile picture: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
  
    public function Progress(){
      try {
        
             $user = Auth::user();
            if (!$user) {
                return response()->json([
                    'status' => 401,
                    'message' => 'Unauthorized: User not authenticated',
                ], 401);
            }
            
           // My progress
           $meetingCount = PrayerLog::where('user_id', $user->id)->count();
           $user_id_prayed = PrayerLog::where('user_id', $user->id)->where('status','Prayed')->count();

           $total_my_prayed_prayer_points = PrayerLog::where('church_id', $user->church_id)
                                    ->where('user_id', $user_id_prayed)
                                    ->where('status','Prayed')
                                    ->count();

        //Interaceessory Army app[records on the basis of church]
        $TotalChurchPrayerMeetings = Meeting::where('church_id', $user->church_id)->count();
        $TotalPrayerWarriors = User::where('church_id',$user->church_id)
        ->where('user_role','Prayer Warrior')->count();
        
         $prayer_points_prayed_count = PrayerLog::where('user_id',$user->id)
                                    ->where('status','Prayed')->count(); 


        //My Prayercell scorecard
        $TotalPrayerWarriorsstatus = User::where('church_id',$user->church_id)
                                         ->where('user_status','Prayer Warrior')->count();
          
        $TotalChampionWarriorsstatus = User::where('church_id',$user->church_id)
                                        ->where('user_status','Champion Warrior')->count();
          
        $TotalEliteWarriorsstatus = User::where('church_id',$user->church_id)
                                        ->where('user_status','Elite Warrior')->count();

        // Prayer Warrior count for the current month
        $CurrentMonthPrayerWarriors = User::where('church_id', $user->church_id)
                                        ->where('user_role', 'Prayer Warrior')
                                        ->whereMonth('created_at', Carbon::now()->month)
                                        ->whereYear('created_at', Carbon::now()->year)
                                        ->count();

        // Prayer Warrior count for the last month
        $LastMonthPrayerWarriors = User::where('church_id', $user->church_id)
                                   ->where('user_role', 'Prayer Warrior')
                                   ->whereMonth('created_at', Carbon::now()->subMonth()->month)
                                   ->whereYear('created_at', Carbon::now()->subMonth()->year)
                                   ->count();
            
        //Last two month
        $LastToLastMonthPrayerWarriors = User::where('church_id', $user->church_id)
                                    ->where('user_role', 'Prayer Warrior')
                                    ->whereMonth('created_at', Carbon::now()->subMonths(2)->month)
                                    ->whereYear('created_at', Carbon::now()->subMonths(2)->year)
                                    ->count();
                                    
                                    
        // Church prayer meetings count for the current month
        $CurrentMonthChurchMeetings = Meeting::where('church_id', $user->church_id)
                                    ->whereMonth('created_at', Carbon::now()->month)
                                    ->whereYear('created_at', Carbon::now()->year)
                                    ->count();

        // Church prayer meetings count for the last month
        $LastMonthChurchMeetings = Meeting::where('church_id', $user->church_id)
                                    ->whereMonth('created_at', Carbon::now()->subMonth()->month)
                                    ->whereYear('created_at', Carbon::now()->subMonth()->year)
                                    ->count();

        // Church prayer meetings count for the last-to-last month
        $LastToLastMonthChurchMeetings = Meeting::where('church_id', $user->church_id)
                            ->whereMonth('created_at', Carbon::now()->subMonths(2)->month)
                            ->whereYear('created_at', Carbon::now()->subMonths(2)->year)
                            ->count();
                            
                            
                            
        //current month prayed points in church             
         $CurrentMonthPrayedPoints = 
            $this->getMonthlyPrayedPoints(PrayerLog::class, $user->church_id, 0);
  

        $LastMonthPrayedPoints = 
            $this->getMonthlyPrayedPoints(PrayerLog::class, $user->church_id, 1);
      

        $LastToLastMonthPrayedPoints = 
            $this->getMonthlyPrayedPoints(PrayerLog::class, $user->church_id, 2) ;
  
            
                            
        return response()->json([
            'status' => 200,
            'message' => 'Progress retrieved successfully',
            'data' => [
                
                'intercessoryarmy' => [
                    
                    'prayer_points_prayed_count'=>$prayer_points_prayed_count,
                    'TotalPrayerWarriors' => $TotalPrayerWarriors,
                    'TotalChurchPrayerMeetings' => $TotalChurchPrayerMeetings,
                 ],
                
                'progress' => [
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'meeting_count' => $meetingCount,
                    'total_my_prayed_prayer_points'=>$total_my_prayed_prayer_points,
                ],
            
                'Prayercellscorecard'=>[
                    
                    'TotalChampionWarriorstatus'=>$TotalChampionWarriorsstatus,
                    'TotalEliteWarriorsstatus'=>$TotalEliteWarriorsstatus,
                    'TotalPrayerWarriorsstatus'=>$TotalPrayerWarriorsstatus,
                    
                    'CurrentMonthPrayerWarriors'=>$CurrentMonthPrayerWarriors,
                    'LastMonthPrayerWarriors'=>$LastMonthPrayerWarriors,
                    'LastToLastMonthPrayerWarriors'=>$LastToLastMonthPrayerWarriors,
                    
                    'CurrentMonthChurchMeetings'=>$CurrentMonthChurchMeetings,
                    'LastMonthChurchMeetings'=>$LastMonthChurchMeetings,
                    'LastToLastMonthChurchMeetings'=>$LastToLastMonthChurchMeetings,
                    
                    'CurrentMonthPrayedPoints'=>$CurrentMonthPrayedPoints,
                    'LastMonthPrayedPoints'=>$LastMonthPrayedPoints,
                    'LastToLastMonthPrayedPoints'=>$LastToLastMonthPrayedPoints,
                ],
            ],
        ], 200);
        
       } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    public function getMonthlyPrayedPoints($table, $churchId, $monthsAgo = 0)
    {
        return $table::where('church_id', $churchId)
            ->where('status', 'Prayed')
            ->whereMonth('created_at', Carbon::now()->subMonths($monthsAgo)->month)
            ->whereYear('created_at', Carbon::now()->subMonths($monthsAgo)->year)
            ->count();
    }
    
    public function Authentication(Request $request)
    {
        try {
         
            // Get the bearer token from the request
            $token = $request->bearerToken();
    
            if (!$token) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token not provided',
                ], 422);
            }
    
            // Find the token
            $accessToken = PersonalAccessToken::findToken($token);
    
            if (!$accessToken) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid token',
                ], 422);
            }
    
            // Retrieve the associated user
            $user = $accessToken->tokenable;
    
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not found',
                ], 422);
            }
    
            // Check if the token is expired
            if ($accessToken->expires_at && now()->gt($accessToken->expires_at)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Token expired',
                ], 200);
            }
    
    
    
       $church_name = $user->church ? $user->church->church_name : null;
        $church_logo = $user->church ? asset($user->church->church_logo) : null;
        
            // Return user details
            return response()->json([
                'status' => 'success',
                'message' => 'User Retrived successfully',
                'data' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'phone_no' => $user->phone_no,
                    'email' => $user->email,
                    'user_role' => $user->user_role,
                    'user_status' => $user->user_status,
                    'profile_photo' => $user->profile_photo ? asset($user->profile_photo) : null,
                    'church_logo' => $church_logo ?? null,
                    'church_name' => $church_name,
                    'profile_photo' => asset($user->profile_photo),
                ],
            ], 200);
    
        } catch (\Throwable $th) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }
}
