<?php

namespace App\Http\Controllers\Api;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Meeting;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Models\PrayerCategory;
use App\Models\PrayerSubCategory;
use App\Models\KeyPrayerPoints;
use App\Models\ChurchPrayerPoint;
use App\Models\ThemePrayerPoint;
use App\Models\PrayerRequest;
use App\Models\Attendance;
use App\Models\PrayerPoint;
use App\Models\PrayerLog;

class MeetingController extends Controller
{
    public function CreateMeeting(Request $request){
        
    try {
            $user = Auth::user();
    
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
    
            $user_id = $user->id;
            $church_id = $user->church_id;
    
            //Validating fields
            $validator = Validator::make($request->all(), [
                'meeting_title' => 'required|string',
                'meeting_date' => ['required', 'date_format:d-m-Y'], 
                'meeting_start_time' => 'required|date_format:H:i',  
                'meeting_end_time' => 'required|date_format:H:i', 
                'mode_of_meeting'=>'required',
                // 'meeting_place'=>'nullable',
                'key_prayer_points'=>'required|integer',
                'prayer_requests'=>'required|integer',
                'church_prayer_points' => 'required', 
                'theme_prayer_points' => 'required',  
            ]);
            
            if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
            
    
            $formattedDate = Carbon::parse($request->meeting_date)->format('Y-m-d');

          // Generate unique Jitsi meeting link
            $uniqueMeetingName = Str::slug($request->meeting_title) . '-' . time();
            $jitsiMeetingLink = "https://meet.jit.si/" . $uniqueMeetingName;
        
            // Creating a new prayer point
            $meeting = new Meeting();
            $meeting->user_id = $user_id;
            $meeting->church_id = $church_id;
            $meeting->meeting_title = $request->meeting_title;
            $meeting->meeting_date = $formattedDate;
            $meeting->meeting_start_time = $request->meeting_start_time;
            $meeting->meeting_end_time = $request->meeting_end_time;
            $meeting->mode_of_meeting = $request->mode_of_meeting;
            $meeting->meeting_place = $request->meeting_place;
            $meeting->key_prayer_points = $request->key_prayer_points;
            $meeting->prayer_requests = $request->prayer_requests;
            $meeting->theme_prayer_sub_category_id = $request->theme_prayer_sub_category_id;
            // $meeting->theme_prayer_points = $request->theme_prayer_points;
            // $meeting->church_prayer_points = $request->church_prayer_points;
            $meeting->theme_prayer_points = json_encode($request->theme_prayer_points, JSON_NUMERIC_CHECK);
            $meeting->church_prayer_points = json_encode($request->church_prayer_points, JSON_NUMERIC_CHECK);
       
            
            // dd($meeting->theme_prayer_points );
        
            $meeting->meeting_link = $jitsiMeetingLink;
            $meeting->save();

            return response()->json([
                'status' => 200,
                'message' => 'Meeting Created successfully',
                'Meeting' => $meeting,
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
       
    }
    
    public function ShowMeeting(){
    try {
          $user = Auth::user();
    
            if (!$user) {
                return response()->json([
                    'status' => 404,
                    'message' => 'User not found',
                ], 404);
            }
            
         $meetings = Meeting::where('user_id',$user->id)
            ->where('church_id',$user->church_id)
            ->where('status', 'Upcoming')
            ->whereDate('meeting_date', '>=', Carbon::today()) 
            ->orderBy('meeting_date', 'asc')
            ->get(['id', 'user_id','church_id','meeting_title','meeting_date','meeting_start_time','meeting_end_time','key_prayer_points', 'prayer_requests','church_prayer_points','theme_prayer_points','mode_of_meeting' ,'meeting_place','prayer_requests','meeting_link','created_at']); // Select required fields

            if ($meetings->isEmpty()) {
                return response()->json([
                    'status' => 404,
                    'message' => 'No Upcoming Meetings Found with this user.',
                ], 404);
            }
            
            // Format the response
            $formattedMeetings = $meetings->map(function ($meeting) {
            
    
                          
        

                $prayerPointIds = !empty($meeting->church_prayer_points)
                    ? json_decode($meeting->church_prayer_points, true)
                    : [];
                if (!is_array($prayerPointIds)) {
                    $prayerPointIds = [$prayerPointIds];
                }
    
                // Retrieve all matching prayer points in one query
                $churchprayerPoints = PrayerPoint::whereIn('id', $prayerPointIds)
                    ->get(['id','prayer_point']);

                
            
                $themePrayerPointIds = !empty($meeting->theme_prayer_points)
                ? json_decode($meeting->theme_prayer_points, true) 
                : [];
                
                  if (!is_array($themePrayerPointIds)) {
                        $themePrayerPointIds = [$themePrayerPointIds];
                    }
    
                $themePrayerPoints = PrayerPoint::whereIn('id', $themePrayerPointIds)
                 ->get(['id', 'prayer_point']);
             
                
                //prayerrequest
                $prayer_category_id_church = PrayerCategory::where('prayer_category_name', 'Pray for my Church')->pluck('id')->first();
               $prayerrequestcount = $meeting->prayer_requests;
                $prayerrequests = PrayerPoint::where('prayer_category_id',$prayer_category_id_church)
                                 ->limit($prayerrequestcount) 
                                  ->get(['id','prayer_point']);
                                  
                                  
                //key_prayer_points       
                $prayer_category_id_prayer_request = PrayerCategory::where('prayer_category_name', 'Prayer Request')->pluck('id')->first();
                $keyprayercount = $meeting->key_prayer_points;
                $keyprayerpoints = PrayerPoint::where('prayer_category_id',$prayer_category_id_prayer_request)
                            ->limit($keyprayercount)
                            ->get(['id','prayer_point']);
   
                return [
                'id' => $meeting->id,
                // 'user_id'=>$meeting->user_id,
                // 'church_id'=>$meeting->church_id,
                'meeting_title' => $meeting->meeting_title ?? '',
                'meeting_date' => $meeting->meeting_date ?? '',
                'meeting_start_time' => $meeting->meeting_start_time ?? '',
                'meeting_end_time' => $meeting->meeting_end_time ?? '',
                'mode_of_meeting' => $meeting->mode_of_meeting ?? '',
                'meeting_place' => $meeting->meeting_place ?? '',
                'meeting_link'=>$meeting->meeting_link ?? '',
                'created_at' => $meeting->created_at ?? '',
                'church_prayer_points' => $churchprayerPoints ?? '',
                'theme_prayer_points'=>$themePrayerPoints ?? '',
                'prayer_requests'=>$prayerrequests ?? '',
                'key_prayer_points'=>$keyprayerpoints ?? '',
            ];
        });

        return response()->json([
            'status' => 200,
            'message' => 'All Meetings retrieved successfully',
            'meetings' => $formattedMeetings,
        ], 200);
        
        } catch (\Exception $e) {
            
            return response()->json([
               'status' => 500,
                'message' => 'Internal Server Error',
                    'error' => $e->getMessage(),
            ], 500);
        }
    }

    
    public function CancleMeeting($id){
    try {
    
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found.'
            ], 404);
        }

        $meeting = Meeting::find($id);

        // Check if the meeting exists
        if (!$meeting) {
            return response()->json([
                'success' => false,
                'message' => 'Meeting not found.'
            ], 404);
        }

        if ($meeting->user_id !== $user->id && $meeting->church_id !== $user->church_id) {
            return response()->json([
                'status' => 403,
                'message' => 'You are not authorized to cancel this meeting',
            ], 403);
        }
        
        //Update status to "Cancelled"
        $meeting->status = 'Cancelled';
        $meeting->save();

        return response()->json([
            'success' => 200,
            'message' => 'Meeting has been canceled successfully.',
             'meeting' => [
                 'id'=>$meeting->id,
                'meeting_title' => $meeting->meeting_title,
                'status' => $meeting->status,
            ],
        ], 200);

        } catch (\Exception $e) {
            // Handle unexpected errors
            return response()->json([
               'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    public function ShowAllCategory()
    {
        try {
            $prayercategory = PrayerCategory::all();
    
            return response()->json([
                'status' => 200,
                'message' => 'Prayer Categories retrieved successfully',
                'prayercategory' =>$prayercategory,

            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    
   public function ShowPrayerFilterPoints(Request $request)
    {
    if ($request->has('prayer_category')) {
        $categoryName = trim($request->prayer_category);


        // Check for 'key prayer points' category
        if ($categoryName === 'Key Prayer Points') {
            $category = PrayerCategory::where('prayer_category_name', 'Key Prayer Points')->first();

            if ($category) {
                $keyprayerpoints = PrayerPoint::where('prayer_category_id', $category->id)->get();

                return response()->json([
                    'status' => 200,
                    'message' => 'Key Prayer points retrieved successfully',
                    'keyprayerpoints' => $keyprayerpoints,
                ], 200);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'Key prayer points category not found',
                ], 404);
            }
        }

        // Check for 'Pray for my Church' category
        elseif ($categoryName === 'Pray for my Church') {
            $category = PrayerCategory::where('prayer_category_name', 'Pray for my Church')->first();

            if ($category) {
                $churchprayerpoints = PrayerPoint::where('prayer_category_id', $category->id)->get();

                return response()->json([
                    'status' => 200,
                    'message' => 'Church Prayer points retrieved successfully',
                    'churchprayerpoints' => $churchprayerpoints,
                ], 200);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'Pray for my Church category not found',
                ], 404);
            }
        }elseif($categoryName === 'Prayer Request') {
            
            $category = PrayerCategory::where('prayer_category_name', 'Prayer Request')->first();

            if ($category) {
                $prayerrequestpoints = PrayerPoint::where('prayer_category_id', $category->id)->get();

                return response()->json([
                    'status' => 200,
                    'message' => 'Prayer Request points retrieved successfully',
                    'prayerrequestpoints' => $prayerrequestpoints,
                ], 200);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'Prayer Request category not found',
                ], 404);
            }
        } elseif ($categoryName === 'Prayer Theme') {
            $category = PrayerCategory::where('prayer_category_name', 'Prayer Theme')->first();

            if ($category) {
                
                if ($request->has('prayer_subcategory')) {
                    $subcategoryName = $request->prayer_subcategory;

                    $subcategory = PrayerSubCategory::where('prayer_sub_cat_name', $subcategoryName)
                        ->where('prayer_category_id', $category->id)
                        ->first();

                    if ($subcategory) {
                        
                        $themeprayerpoints = PrayerPoint::where('prayer_sub_category_id', $subcategory->id)->get();

                        return response()->json([
                            'status' => 200,
                            'message' => 'Prayer Theme and points retrieved successfully',
                            'themeprayerpoints' => $themeprayerpoints,
                        ], 200);
                    } else {
                        return response()->json([
                            'status' => 404,
                            'message' => 'Subcategory not found or does not belong to the selected Prayer Theme',
                        ], 404);
                    }
                } 
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'Prayer Theme category not found',
                ], 404);
            }
        }

        return response()->json([
            'status' => 404,
            'message' => 'Category not found',
        ], 404);
    }

    return response()->json([
        'status' => 400,
        'message' => 'Missing prayer_category parameter',
     ], 400);
    }
    
    
    public function StartMeeting($id, Request $request)
    {
        try {
        $user = auth()->user();
        
         $meeting = Meeting::where('id', $id)
                          ->where('user_id', $user->id) 
                          ->first();
        

        if (!$meeting) {
            return response()->json([
                'status' => 400,
                'message' => 'The Meeting not found.',
            ], 400);
        }
        
        // Check if attendance already exists
        $attendance = Attendance::where('meeting_id', $meeting->id)
                                 ->where('meeting_started_by', $user->id)
                                 ->first();

        // If attendance exists, do nothing (user can join again)
        if (!$attendance) {
            
            Attendance::create([
                'meeting_id' => $meeting->id,
                'meeting_started_by' => $user->id,
                'church_id' => $user->church_id, 
            ]);
        }
        
        // Get Church Prayer Points
        $churchPrayerPointIds = !empty($meeting->church_prayer_points) ? json_decode($meeting->church_prayer_points, true) : [];
        $churchPrayerPoints = PrayerPoint::whereIn('id', $churchPrayerPointIds)
                                      
                                       ->get(['id','prayer_category_id', 'prayer_point'])
                                       ->map(function($prayerPoint) {
                                           $prayerPoint->source = 'ChurchPrayerPoint'; 
                                           return $prayerPoint;
                                       });

        // Get Theme Prayer Points
        $themePrayerPointIds = !empty($meeting->theme_prayer_points) ? json_decode($meeting->theme_prayer_points, true) : [];
        $themePrayerPoints = PrayerPoint::whereIn('id', $themePrayerPointIds)
                                     ->get(['id','prayer_category_id','prayer_sub_category_id', 'prayer_point'])
                                     ->map(function($prayerPoint) {
                                         $prayerPoint->source = 'ThemePrayerPoint'; 
                                         return $prayerPoint;
                                     });

        // Get Key Prayer Points
        $keyPrayerCount = $meeting->key_prayer_points;
        $keyPrayerPoints = PrayerPoint::limit($keyPrayerCount)
                                  ->get(['id', 'prayer_category_id','prayer_point'])
                                  ->map(function($prayerPoint) {
                                      $prayerPoint->source = 'KeyPrayerPoints'; 
                                      return $prayerPoint;
                                  });

        // Get Prayer Requests
        $prayerrequestcount = $meeting->prayer_requests;
        $Prayerrequests = PrayerPoint::limit($prayerrequestcount)
                               ->get(['id','prayer_category_id','prayer_point'])
                               ->map(function($prayerPoint) {
                                   $prayerPoint->source = 'PrayerRequest'; 
                                   return $prayerPoint;
                               });
                                          
        $allPrayerPoints = collect();  
         $allPrayerPoints = $churchPrayerPoints->merge($themePrayerPoints)
                                                ->merge($keyPrayerPoints)
                                                ->merge($Prayerrequests);
                                                
         
        // Fetch all users except the authenticated user
        $all_warriors = User::where('id', '!=', $user->id)
                            ->get(['id', 'first_name', 'last_name']);

    
        $warriorsList = $all_warriors->map(function ($warrior) {
            return [
                'id' => $warrior->id,
                'name' => $warrior->first_name . ' ' . $warrior->last_name
            ];
        });
         
         $meetingDetails = [
            'id' => $meeting->id,
            'meeting_title' => $meeting->meeting_title,
            'status'=>$meeting->status,
            'prayer_points' => $allPrayerPoints, 
            'warriors' => $warriorsList,
         ];

        return response()->json([
            'status' => 200,
            'message' => 'Meeting started successfully',
            'meeting' => $meetingDetails,
        ], 200);
        
        } catch (\Exception $e) {
            \Log::error('Error in AddPrayerRequest function: ' . $e->getMessage()); 
            return response()->json([
                'status' => 500,
                 'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    
    public function JoinMeetingParticipants($id)
    {
        // Validate if meeting exists
        $meeting = Meeting::find($id);

        if (!$meeting) {
            return response()->json(['message' => 'Meeting not found'], 404);
        }
    
        $attendance = Attendance::where('meeting_id', $id)->first();

        if (!$attendance || empty($attendance->user_ids)) {
            return response()->json(['message' => 'No participants found'], 404);
        }

        // Decode user_ids JSON
        $userIds = json_decode($attendance->user_ids, true);

        if (!is_array($userIds) || empty($userIds)) {
            return response()->json(['message' => 'Invalid participants data'], 400);
        }

        // Fetch user details
        $participants = User::whereIn('id', $userIds)->select('id', 'first_name','last_name')->get();

        return response()->json([
            'meeting_id' => $meeting->id,
            'participants' => $participants
        ]);
    }
    
      public function UpdateChurchPoint($id, Request $request)
      {
        try {
            $meeting = Meeting::find($id);
    
            if (!$meeting) {
                return response()->json(['message' => 'Meeting not found'], 404);
            }
    
            // Get and validate the input for church prayer points
            $prayerPoints = json_decode($request->church_prayer_points, true);
    
            // Validate input
            $validator = Validator::make(['church_prayer_points' => $prayerPoints], [
                'church_prayer_points'   => 'required|array',
                'church_prayer_points.*' => 'integer',
            ]);
    
            if ($validator->fails()) {
                $errors = $validator->errors()->all();
    
                return response()->json([
                    'status' => 422,
                    'message' => $errors,
                ], 422);
            }
    
            // Update the church prayer points
            $meeting->church_prayer_points = json_encode($prayerPoints, JSON_NUMERIC_CHECK);
    
            // Save the changes
            $meeting->save();
    
            return response()->json([
                'status' => 200,
                'message' => 'Church Prayer Points updated successfully',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in UpdateChurchPoint function: ' . $e->getMessage());
    
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

    public function UpdateThemePoint($id, Request $request)
    {
        try {
            $meeting = Meeting::find($id);
    
            if (!$meeting) {
                return response()->json(['message' => 'Meeting not found'], 404);
            }
    
            // Get and validate the input for theme prayer points
            $prayerPoints = json_decode($request->theme_prayer_points, true);
    
            // Validate input
            $validator = Validator::make(['theme_prayer_points' => $prayerPoints], [
                'theme_prayer_points'   => 'required|array',
                'theme_prayer_points.*' => 'integer',
            ]);
    
            if ($validator->fails()) {
                $errors = $validator->errors()->all();
    
                return response()->json([
                    'status' => 422,
                    'message' => $errors,
                ], 422);
            }
    
            // Update the theme prayer points
            $meeting->theme_prayer_points = json_encode($prayerPoints, JSON_NUMERIC_CHECK);
    
            // Save the changes
            $meeting->save();
    
            return response()->json([
                'status' => 200,
                'message' => 'Theme Prayer Points updated successfully',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in UpdateThemePoint function: ' . $e->getMessage()); // Log error for debugging
    
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }


    
    public function PrayerPointPrayed($id, Request $request)
    {
        try {
            $meeting = Meeting::find($id);
    
            if (!$meeting) {
                return response()->json(['message' => 'Meeting not found'], 404);
            }
            
         
            // Validate request inputs
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|integer|exists:users,id',
                'prayer_points_id' => 'required|integer|exists:prayer_points,id',
            ]);
    
           if ($validator->fails()) {
                $errors = $validator->errors()->all(); 
            
                return response()->json([
                    'status' => 422,
                    'message' => $errors, 
                ], 422);
            }
            
            // Retrieve user details
            $user = User::find($request->user_id);
            if (!$user) {
                return response()->json(['message' => 'User not found'], 404);
            }

            // Store the prayer log entry
            $prayerLog = new PrayerLog();
            $prayerLog->church_id = $user->church_id;
            $prayerLog->meeting_id = $meeting->id;
            $prayerLog->user_id = $request->user_id;
            $prayerLog->prayer_points_id = $request->prayer_points_id;
            $prayerLog->status = "Prayed";
            $prayerLog->save();
    
            return response()->json([
                'status' => 200,
                'message' => 'Prayer log recorded successfully',
                // 'prayer_log' => $prayerLog
            ]);
    
        } catch (\Exception $e) {
             \Log::error('Error in AddPrayerRequest function: ' . $e->getMessage()); 
            return response()->json([
                'status' => 500,
                 'message' => 'Internal Server Error',
            ], 500);
        }
    }
    
    
    public function PrayerLog($id){
        
         try {
            $meeting = Meeting::find($id);
    
            if (!$meeting) {
                return response()->json(['message' => 'Meeting not found'], 404);
            }
          
            $prayer_logs = PrayerLog::with('user','prayerPoint')->where('meeting_id', $meeting->id)->get();
           
    
            $filtered_logs = $prayer_logs->map(function ($log) {
                    return [
                        'id' => $log->id, 
                        'user_name' => $log->user ? $log->user->first_name . ' ' . $log->user->last_name : 'Unknown User',
                        'prayer_point' => $log->prayerPoint ? $log->prayerPoint->prayer_point : '',
                        'created_at' => $log->created_at->format('g:i A'),
                    ];
                });

            return response()->json([
                'status' => 200,
                'message' => 'Prayer log retrived successfully',
                'data' => $filtered_logs,
            ]);
    
        } catch (\Exception $e) {
             \Log::error('Error in UpdateThemePoint function: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Internal Server Error',
            ], 500);
        }
    }

}
